'use strict';

exports.__esModule = true;

var _createClass = function () { function defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } } return function (Constructor, protoProps, staticProps) { if (protoProps) defineProperties(Constructor.prototype, protoProps); if (staticProps) defineProperties(Constructor, staticProps); return Constructor; }; }();

var _const = require('../../const');

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

/**
 * Rectangle object is an area defined by its position, as indicated by its top-left corner
 * point (x, y) and by its width and its height.
 *
 * @class
 * @memberof PIXI
 */
var Rectangle = function () {
  /**
   * @param {number} [x=0] - The X coordinate of the upper-left corner of the rectangle
   * @param {number} [y=0] - The Y coordinate of the upper-left corner of the rectangle
   * @param {number} [width=0] - The overall width of this rectangle
   * @param {number} [height=0] - The overall height of this rectangle
   */
  function Rectangle() {
    var x = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;
    var y = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    var width = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 0;
    var height = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;

    _classCallCheck(this, Rectangle);

    /**
     * @member {number}
     * @default 0
     */
    this.x = Number(x);

    /**
     * @member {number}
     * @default 0
     */
    this.y = Number(y);

    /**
     * @member {number}
     * @default 0
     */
    this.width = Number(width);

    /**
     * @member {number}
     * @default 0
     */
    this.height = Number(height);

    /**
     * The type of the object, mainly used to avoid `instanceof` checks
     *
     * @member {number}
     * @readOnly
     * @default PIXI.SHAPES.RECT
     * @see PIXI.SHAPES
     */
    this.type = _const.SHAPES.RECT;
  }

  /**
   * returns the left edge of the rectangle
   *
   * @member {number}
   */


  /**
   * Creates a clone of this Rectangle
   *
   * @return {PIXI.Rectangle} a copy of the rectangle
   */
  Rectangle.prototype.clone = function clone() {
    return new Rectangle(this.x, this.y, this.width, this.height);
  };

  /**
   * Copies another rectangle to this one.
   *
   * @param {PIXI.Rectangle} rectangle - The rectangle to copy.
   * @return {PIXI.Rectangle} Returns itself.
   */


  Rectangle.prototype.copy = function copy(rectangle) {
    this.x = rectangle.x;
    this.y = rectangle.y;
    this.width = rectangle.width;
    this.height = rectangle.height;

    return this;
  };

  /**
   * Checks whether the x and y coordinates given are contained within this Rectangle
   *
   * @param {number} x - The X coordinate of the point to test
   * @param {number} y - The Y coordinate of the point to test
   * @return {boolean} Whether the x/y coordinates are within this Rectangle
   */


  Rectangle.prototype.contains = function contains(x, y) {
    if (this.width <= 0 || this.height <= 0) {
      return false;
    }

    if (x >= this.x && x < this.x + this.width) {
      if (y >= this.y && y < this.y + this.height) {
        return true;
      }
    }

    return false;
  };

  /**
   * Pads the rectangle making it grow in all directions.
   *
   * @param {number} paddingX - The horizontal padding amount.
   * @param {number} [paddingY] - The vertical padding amount.
   */


  Rectangle.prototype.pad = function pad(paddingX, paddingY) {
    paddingX = paddingX || 0;
    paddingY = paddingY || (paddingY !== 0 ? paddingX : 0);

    this.x -= paddingX;
    this.y -= paddingY;

    this.width += paddingX * 2;
    this.height += paddingY * 2;
  };

  /**
   * Fits this rectangle around the passed one.
   *
   * @param {PIXI.Rectangle} rectangle - The rectangle to fit.
   */


  Rectangle.prototype.fit = function fit(rectangle) {
    var x1 = Math.max(this.x, rectangle.x);
    var x2 = Math.min(this.x + this.width, rectangle.x + rectangle.width);
    var y1 = Math.max(this.y, rectangle.y);
    var y2 = Math.min(this.y + this.height, rectangle.y + rectangle.height);

    this.x = x1;
    this.width = Math.max(x2 - x1, 0);
    this.y = y1;
    this.height = Math.max(y2 - y1, 0);
  };

  /**
   * Enlarges this rectangle to include the passed rectangle.
   *
   * @param {PIXI.Rectangle} rectangle - The rectangle to include.
   */


  Rectangle.prototype.enlarge = function enlarge(rectangle) {
    var x1 = Math.min(this.x, rectangle.x);
    var x2 = Math.max(this.x + this.width, rectangle.x + rectangle.width);
    var y1 = Math.min(this.y, rectangle.y);
    var y2 = Math.max(this.y + this.height, rectangle.y + rectangle.height);

    this.x = x1;
    this.width = x2 - x1;
    this.y = y1;
    this.height = y2 - y1;
  };

  /**
   * Enlarges rectangle that way its corners lie on grid
   *
   * @param {number} [resolution=1] resolution
   * @param {number} [eps=0.001] precision
   */


  Rectangle.prototype.ceil = function ceil() {
    var resolution = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
    var eps = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.001;

    var x2 = Math.ceil((this.x + this.width - eps) * resolution) / resolution;
    var y2 = Math.ceil((this.y + this.height - eps) * resolution) / resolution;

    this.x = Math.floor((this.x + eps) * resolution) / resolution;
    this.y = Math.floor((this.y + eps) * resolution) / resolution;

    this.width = x2 - this.x;
    this.height = y2 - this.y;
  };

  _createClass(Rectangle, [{
    key: 'left',
    get: function get() {
      return this.x;
    }

    /**
     * returns the right edge of the rectangle
     *
     * @member {number}
     */

  }, {
    key: 'right',
    get: function get() {
      return this.x + this.width;
    }

    /**
     * returns the top edge of the rectangle
     *
     * @member {number}
     */

  }, {
    key: 'top',
    get: function get() {
      return this.y;
    }

    /**
     * returns the bottom edge of the rectangle
     *
     * @member {number}
     */

  }, {
    key: 'bottom',
    get: function get() {
      return this.y + this.height;
    }

    /**
     * A constant empty rectangle.
     *
     * @static
     * @constant
     */

  }], [{
    key: 'EMPTY',
    get: function get() {
      return new Rectangle(0, 0, 0, 0);
    }
  }]);

  return Rectangle;
}();

exports.default = Rectangle;
//# sourceMappingURL=Rectangle.js.map